function [rms_errors, cdx_new, tranche_rms] = RMSE_tranche_mispricing2(CDX, start_date_num, end_date_num, maturities, RMSE_method, ...
                                                                       display, CDX2, CDX3)
% --------------------------------------------------------------------------------------------------
% For each date, calculate square root of average squared tranche pricing error between 'start_date'
% and 'end_date'. Note that model-implied tranche prices need to be calculated beforehand. This
% function simply adds up all squared pricing errors for each day
% --------------------------------------------------------------------------------------------------
% CDX               ... credit index structure (see 'all_steps_in_a_row.m')
% start_date_num    ... datenum of start date
% end_date_num      ... datenum of end date
% maturities        ... indicator vector which maturities to include
% RMSE_method       ... which definition of pricing error to use
%                       0 ... absolute RMSE
%                       1 ... relative RMSE
%                       2 ... RMSE weighted by inverse of tranche bid-ask
%                       3 ... RMSE weighted by tranche notional size
%                       4 ... robust version of 1, set error = 0 if model-price between mid-price of CDX and CDX2
%                       5 ... same as 4, but with three CDX structures
% display           ... indicator, whether to display average RMSE
% CDX2              ... second CDX structure for robust RMSE calculation (method 4)
% CDX3              ... third CDX structure for robust RMSE calculation (method 5)
% --------------------------------------------------------------------------------------------------
% sample call: RMSE_tranche_mispricing2(cdx_tv_monthly, cdx_tv_monthly.dates{1}(1), cdx_tv_monthly.dates{1}(end), [1 0 0], 4, 1, CDX_NA_IG)
%              RMSE_tranche_mispricing2(cdx_tv_monthly10, cdx_tv_monthly10.dates{1}(1), cdx_tv_monthly10.dates{1}(end), [1 1 1], 4, 1, CDX_NA_IG)
% --------------------------------------------------------------------------------------------------

% Sum up squared pricing erros for all desired maturities
for i=1:length(maturities)
    if (maturities(i) == 1)
        % Determine relevant date range
        start_pos = find(CDX.dates{i} >= start_date_num, 1, 'first');
        end_pos = find(CDX.dates{i} <= end_date_num, 1, 'last');
        used_dates = CDX.dates{i}(start_pos:end_pos);
        
        % Allocate memory for output
        if (i==1)
            date_range = CDX.dates{i}(start_pos:end_pos);
            num_dates  = end_pos - start_pos + 1;
            num_prices = zeros(num_dates, 1);
            errors = zeros(num_dates, 1);
        end
        
        % Extract model_implied and market prices of tranches
        market_price = CDX.tranche_market_price_mid{i}(start_pos:end_pos,:);
        market_upfront = CDX.tranche_market_upfront_mid{i}(start_pos:end_pos,:);
        model_price = CDX.tranche_model_price{i}(start_pos:end_pos,:);
        model_upfront = CDX.tranche_model_upfront{i}(start_pos:end_pos,:);
        
        % Determine position where to add pricing errors
        [trash, ai, bi] = intersect(date_range, CDX.dates{i}(start_pos:end_pos));
        if (isempty(ai))
            continue;
        end
        
        % Add squared pricing errors
        num_prices(ai) = num_prices(ai) + size(market_price, 2);
        if (RMSE_method == 1)          % use relative RMSE
            rel_error_price = (market_price - model_price)./market_price;
            rel_error_upfront = (market_upfront - model_upfront)./max(1e-10, market_upfront);
            tranche_rms = (rel_error_price + rel_error_upfront);
            rel_error_price = rel_error_price.^2;
            rel_error_upfront = rel_error_upfront.^2;
            % Cap relative error at 200%
            rel_error_price = min(2, rel_error_price);
            rel_error_upfront = min(2, rel_error_upfront);
            rel_error_price = sum(rel_error_price,2);
            rel_error_upfront = sum(rel_error_upfront,2);
            errors(ai) = errors(ai) + rel_error_price(bi) + rel_error_upfront(bi);
        elseif (RMSE_method == 0)      % use absolute RMSE
            abs_price_error = market_price - model_price;
            abs_upfront_error = market_upfront - model_upfront;
            tranche_rms = (abs_price_error + abs_upfront_error);
            errors(ai) = errors(ai) + sum(abs_price_error(bi,:).^2,2) + sum(abs_upfront_error(bi,:).^2,2);
        elseif (RMSE_method == 2)      % use bid-ask spread weighted RMSE on December 5, 2005
            abs_price_error = market_price - model_price;
            abs_upfront_error = market_upfront - model_upfront;
            tranche_rms = (abs_price_error + abs_upfront_error);
            abs_price_error = abs_price_error ./ repmat([0.8 6.8 5.3 3.0 1.0], size(abs_price_error,2), 1);
            abs_upfront_error = abs_upfront_error ./ repmat([0.8 6.8 5.3 3.0 1.0], size(abs_price_error,2), 1);
            % Cap error at 5 times bid-ask spread
            abs_price_error = min(5, abs_price_error);
            abs_upfront_error = min(5, abs_upfront_error);
            errors(ai) = errors(ai) + sum(abs_price_error(bi,:).^2,2) + sum(abs_upfront_error(bi,:).^2,2);
        elseif (RMSE_method == 3)    % capital weighted pricing error
            price_error = market_price - model_price;
            upfront_error = market_upfront - model_upfront;
            tranche_rms = (price_error + upfront_error);
            weights = repmat(diff([0 CDX.cut_offs]), size(price_error,1), 1) .* market_price;
            price_error = price_error .* weights;
            weights2 = repmat(diff([0 CDX.cut_offs]), size(upfront_error, 1), 1) .* market_upfront * 1e6 / 5;
            upfront_error = upfront_error .* weights2;           
            errors(ai) = errors(ai) + sum(price_error(bi,:),2) + sum(upfront_error(bi,:),2);
        elseif (RMSE_method == 4)   % robust version of 1, set error = 0 if model-price between mid-price of CDX and CDX2
            rel_error_price = (market_price - model_price)./market_price;
            rel_error_upfront = (market_upfront - model_upfront)./max(1e-10, market_upfront);
            
            % Determine relevant date range for CDX2
            start_pos2 = find(CDX2.dates{i} >= start_date_num, 1, 'first');
            end_pos2 = find(CDX2.dates{i} <= end_date_num, 1, 'last');
            used_range2 = (start_pos2:end_pos2);
            used_dates2 = CDX2.dates{i}(start_pos2:end_pos2);
            [trash, used_pos_cdx, used_pos_cdx2] = intersect_sorted(used_dates, used_dates2);
            used_range2 = used_range2(used_pos_cdx2);
            used_dates2 = used_dates2(used_pos_cdx2);

            % Extract market price of tranches (mid, bid and ask)
            market_price_mid = CDX2.tranche_market_price_mid{i}(used_range2,:);
            market_upfront_mid = CDX2.tranche_market_upfront_mid{i}(used_range2,:);
            rel_error_price2 = (market_price_mid - model_price(used_pos_cdx,:))./market_price_mid;
            rel_error_upfront2 = (market_upfront_mid - model_upfront(used_pos_cdx,:))./max(1e-10, market_upfront_mid);

            % Use minimum of relative errors for CDX and CDX2
            price_error_lower = (abs(rel_error_price(used_pos_cdx,:)) > abs(rel_error_price2));
            upfront_error_lower = (abs(rel_error_upfront(used_pos_cdx,:)) > abs(rel_error_upfront2));
            
            % Check if CDX price outside of CDX2 bid-ask range
            for j=1:length(used_dates2)
                % Set RMSE to minumum
                cdx_pos = used_pos_cdx(j);
                rel_error_price(cdx_pos, price_error_lower(j,:)) = rel_error_price2(j, price_error_lower(j,:));
                rel_error_upfront(cdx_pos, upfront_error_lower(j,:)) = rel_error_upfront2(j, upfront_error_lower(j,:));
                
                % Check criterion if model_price between mid-price of CDX and CDX2
                inbetween = ((market_price(cdx_pos,:) < model_price(j,:)) & (model_price(j,:) < market_price_mid(j,:))) | ...
                            ((market_price(cdx_pos,:) > model_price(j,:)) & (model_price(j,:) > market_price_mid(j,:))) | ...
                            ((market_upfront(cdx_pos,:) < model_upfront(j,:)) & (model_upfront(j,:) < market_upfront_mid(j,:))) | ...
                            ((market_upfront(cdx_pos,:) > model_upfront(j,:)) & (model_upfront(j,:) > market_upfront_mid(j,:)));
                set_to_zero = find(inbetween);
                rel_error_price(cdx_pos, set_to_zero) = 0;
                rel_error_upfront(cdx_pos, set_to_zero) = 0;
            end
            
            % Add up errors
            tranche_rms = (rel_error_price + rel_error_upfront);
            rel_error_price = rel_error_price.^2;
            rel_error_upfront = rel_error_upfront.^2;
            % Cap relative error at 200%
            rel_error_price = min(2, rel_error_price);
            rel_error_upfront = min(2, rel_error_upfront);
            errors(ai) = errors(ai) + sum(rel_error_price(bi,:), 2) + sum(rel_error_upfront(bi,:), 2);
        elseif (RMSE_method == 5)   % robust version of 1, set error = 0 if model-price between mid-price of CDX, CDX2 or CDX3
            rel_error_price = (market_price - model_price)./market_price;
            rel_error_upfront = (market_upfront - model_upfront)./max(1e-10, market_upfront);
            
            % Determine relevant date range for CDX2
            start_pos2 = find(CDX2.dates{i} >= start_date_num, 1, 'first');
            end_pos2 = find(CDX2.dates{i} <= end_date_num, 1, 'last');
            used_range2 = (start_pos2:end_pos2);
            used_dates2 = CDX2.dates{i}(start_pos2:end_pos2);
            [trash, used_pos_cdx, used_pos_cdx2] = intersect_sorted(used_dates, used_dates2);
            used_range2 = used_range2(used_pos_cdx2);
            used_dates2 = used_dates2(used_pos_cdx2);
            
            % Extract market price of tranches (mid, bid and ask) for CDX2
            market_price_mid = CDX2.tranche_market_price_mid{i}(used_range2,:);
            market_upfront_mid = CDX2.tranche_market_upfront_mid{i}(used_range2,:);
            rel_error_price2 = (market_price_mid - model_price(used_pos_cdx,:))./market_price_mid;
            rel_error_upfront2 = (market_upfront_mid - model_upfront(used_pos_cdx,:))./max(1e-10, market_upfront_mid);
                                    
            % Use minimum of relative errors for CDX and CDX2
            price_error_lower = (abs(rel_error_price(used_pos_cdx,:)) > abs(rel_error_price2));
            upfront_error_lower = (abs(rel_error_upfront(used_pos_cdx,:)) > abs(rel_error_upfront2));            
            
            % Check if CDX price outside of CDX2 bid-ask range
            for j=1:length(used_dates2)
                % Set RMSE to minumum
                cdx_pos = used_pos_cdx(j);
                rel_error_price(cdx_pos, price_error_lower(j,:)) = rel_error_price2(j, price_error_lower(j,:));
                rel_error_upfront(cdx_pos, upfront_error_lower(j,:)) = rel_error_upfront2(j, upfront_error_lower(j,:));
                
                % Check criterion if model_price between mid-price of CDX and CDX2
                inbetween = ((market_price(cdx_pos,:) < model_price(j,:)) & (model_price(j,:) < market_price_mid(j,:))) | ...
                            ((market_price(cdx_pos,:) > model_price(j,:)) & (model_price(j,:) > market_price_mid(j,:))) | ...
                            ((market_upfront(cdx_pos,:) < model_upfront(j,:)) & (model_upfront(j,:) < market_upfront_mid(j,:))) | ...
                            ((market_upfront(cdx_pos,:) > model_upfront(j,:)) & (model_upfront(j,:) > market_upfront_mid(j,:)));
                set_to_zero = find(inbetween);
                rel_error_price(cdx_pos, set_to_zero) = 0;
                rel_error_upfront(cdx_pos, set_to_zero) = 0;
            end
            
            %%%%%%%%%%%%%%%% Repeat steps for CDX3
            
            % Determine relevant date range for CDX3
            start_pos3 = find(CDX3.dates{i} >= start_date_num, 1, 'first');
            end_pos3 = find(CDX3.dates{i} <= end_date_num, 1, 'last');
            used_range3 = (start_pos3:end_pos3);
            used_dates3 = CDX3.dates{i}(start_pos3:end_pos3);
            [trash, used_pos_cdx, used_pos_cdx3] = intersect_sorted(used_dates, used_dates3);
            used_range3 = used_range3(used_pos_cdx3);
            used_dates3 = used_dates3(used_pos_cdx3);
            
            % Extract market price of tranches (mid, bid and ask) for CDX3
            market_price_mid = CDX3.tranche_market_price_mid{i}(used_range3,:);
            market_upfront_mid = CDX3.tranche_market_upfront_mid{i}(used_range3,:);
            rel_error_price3 = (market_price_mid - model_price(used_pos_cdx,:))./market_price_mid;
            rel_error_upfront3 = (market_upfront_mid - model_upfront(used_pos_cdx,:))./max(1e-10, market_upfront_mid);
                                    
            % Use minimum of relative errors for CDX and CDX3
            price_error_lower = (abs(rel_error_price(used_pos_cdx,:)) > abs(rel_error_price3));
            upfront_error_lower = (abs(rel_error_upfront(used_pos_cdx,:)) > abs(rel_error_upfront3));            
            
            % Check if CDX price outside of CDX3 bid-ask range
            for j=1:length(used_dates3)
                % Set RMSE to minumum
                cdx_pos = used_pos_cdx(j);
                rel_error_price(cdx_pos, price_error_lower(j,:)) = rel_error_price3(j, price_error_lower(j,:));
                rel_error_upfront(cdx_pos, upfront_error_lower(j,:)) = rel_error_upfront3(j, upfront_error_lower(j,:));
                
                % Check criterion if model_price between mid-price of CDX and CDX3
                inbetween = ((market_price(cdx_pos,:) < model_price(j,:)) & (model_price(j,:) < market_price_mid(j,:))) | ...
                            ((market_price(cdx_pos,:) > model_price(j,:)) & (model_price(j,:) > market_price_mid(j,:))) | ...
                            ((market_upfront(cdx_pos,:) < model_upfront(j,:)) & (model_upfront(j,:) < market_upfront_mid(j,:))) | ...
                            ((market_upfront(cdx_pos,:) > model_upfront(j,:)) & (model_upfront(j,:) > market_upfront_mid(j,:)));
                set_to_zero = find(inbetween);
                rel_error_price(cdx_pos, set_to_zero) = 0;
                rel_error_upfront(cdx_pos, set_to_zero) = 0;
            end

            %%%%%%%%%%%%%%%% Add up errors
            tranche_rms = (rel_error_price + rel_error_upfront);
            rel_error_price = rel_error_price.^2;
            rel_error_upfront = rel_error_upfront.^2;
            % Cap relative error at 200%
            rel_error_price = min(2, rel_error_price);
            rel_error_upfront = min(2, rel_error_upfront);
            errors(ai) = errors(ai) + sum(rel_error_price(bi,:), 2) + sum(rel_error_upfront(bi,:), 2);
        end
    end
end

% Calculate RMS-errors and add to CDX structure
if (RMSE_method ~= 3)
    rms_errors = sqrt(errors ./ num_prices);
else
    rms_errors = errors;
end
if (~isfield(CDX, 'rmse'))
    CDX.rmse = zeros(length(CDX.dates{1}), 1);
end
start_pos = find(CDX.dates{1} >= start_date_num, 1, 'first');
end_pos = find(CDX.dates{1} <= end_date_num, 1, 'last');
CDX.rmse_tranches(start_pos:end_pos,1) = rms_errors; 

% Display results
if (RMSE_method ~= 3)
    avg_rms_error = sqrt(sum(rms_errors.^2) / length(rms_errors));
else
    avg_rms_error = mean(rms_errors);
end 
texts = {'Average absolute RMSE (in bps): ' 'Average relative RMSE (fraction): ' ...
         'Liquidity weighted RMSE: ' 'Mean capital weighted error: ' 'Robust relative RMSE (fraction): '};
if (display == 1)
    disp([texts{RMSE_method+1} num2str(avg_rms_error)]);
end
cdx_new = CDX;
